# '==================================================================================================================================================================
# 'Disclaimer
# 'The sample scripts are not supported under any SolarWinds support program or service. 
# 'The sample scripts are provided AS IS without warranty of any kind. 
# 'SolarWinds further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# 'The entire risk arising out of the use or performance of the sample scripts and documentation stays with you. 
# 'In no event shall SolarWinds or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# '(including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# 'arising out of the use of or inability to use the sample scripts or documentation.
# '==================================================================================================================================================================

Param (
    [string]$verbose = ""
)

function setupLogging() {
    $script:logFile = "C:\SolarWinds\TakeControlStateDetectionLog.txt"
    $script:logFolder = Split-Path $logFile

    $logFolderExists = Test-Path $logFile
    $logFileExists = Test-Path $logFile

    If ($logFolderExists -eq $false) {
        New-Item -ItemType "directory" -Path "C:\SolarWinds\"
    }
    
    If ($logFileExists -eq $true) {
        Remove-Item $logfile -ErrorAction SilentlyContinue
    }
}

function validateUserInput() {
    If ($verbose.ToLower() -eq "y") {
        $script:verboseMode = $true
        writeToLog V "You have defined to have the script output the verbose log entries."
    } Else {
        $script:verboseMode = $false
        writeToLog I "Will output logs in regular mode."
    }

    writeToLog I "Completed validating user defined variables."
}

function validateFileLocks() {
    writeToLog V ("Started running {0} function." -f $MyInvocation.MyCommand)

    $script:installLocation = "C:\Program Files (x86)\Take Control Agent\"
    $script:prgDataLocation = "C:\ProgramData\GetSupportService_LOGICnow\"

    $script:installLockFile = $installLocation + "*installing.lock"
    $script:uninstallLockFile = $installLocation + "*uninstalling.lock"
    $script:lockPresent = $false

    writeToLog V "lockPresent variable is set to: $lockPresent"
    writeToLog V "Testing if _installing.lock is present on the device."
    writeToLog V "Testing the following path (including wildcard):`r`n$installLockFile"

    $installLockExist = Test-Path $installLockFile

    If ($installLockExist -eq $true) {
        writeToLog W "Detected _installing.lock on the device."
        $script:installLockCreation = (Get-Item $installLockFile).LastWriteTime.DateTime
        writeToLog W "This was created at: $installLockCreation"

        $script:lockPresent = $true
    } Else {
        writeToLog I "_installing.lock was not found on the device."
    }

    writeToLog V "Now testing to see if _uninstalliing.lock is present on the device."
    writeToLog V "Testing the following path (including wildcard):`r`n$uninstallLockFile"

    $uninstallLockExist = Test-Path $uninstallLockFile

    If ($uninstallLockExist -eq $true) {
        writeToLog W "Detected _uninstalling.lock on the device."
        $script:uninstallLockCreation = (Get-Item $uninstallLockFile).LastWriteTime.DateTime
        writeToLog W "This was created at: $uninstallLockCreation"

        $script:lockPresent = $true
    } Else {
        writeToLog I "_uninstalling.lock was not found on the device."
    }

    writeToLog V "lockPresent evaluation complete."
    writeToLog V "New state of the lockPresent variable after evaluation: $lockPresent"

    writeToLog V ("Completed running {0} function." -f $MyInvocation.MyCommand)
}

function validateEXEs() {
    writeToLog V ("Started running {0} function." -f $MyInvocation.MyCommand)

    $script:exeMissing = $false

    writeToLog V "exe variable is set to: $exeMissing"
    writeToLog V "Checking to see if the device has the all the required executibles present."

    $exeList = @('BASupClpHlp.exe', 'BASupClpPrg.exe', 'BASupConHelper.exe', 'BASupRegEditHlpr.exe', 'BASupSrvc.exe', 'BASupSrvcCnfg.exe', 'BASupSrvcUpdater.exe', 'BASupSysInf.exe', 'BASupSysShell.exe', 'BASupSysShell64.exe', 'BASupTSHelper.exe', 'BASupUnElev.exe', 'BAVideoChat.exe', 'uninstall.exe')

    foreach ($exe in $exeList) {
        If ((Test-Path $installLocation$exe) -eq $false) {
            writeToLog W "Did not detect $exe on the device."
            $script:exeMissing = $true
        } ElseIf ((Test-Path $installLocation$exe) -eq $true) {
            writeToLog V "Successfully detected $exe on the device."
        }
    }

    # Doing a detection on uninstall again.
    # If present and removal required, we can push uninstall.
    $script:uninstallLocation = $installLocation + "uninstall.exe"

    writeToLog V "Testing uninstall.exe with the following path:`r`n$uninstallLocation"

    If ((Test-Path $uninstallLocation) -eq $false) {
        writeToLog V "Did not detect uninstall.exe on the device."
        $script:uninstallPresent = $false
    } Else {
        writeToLog V "Successfully detected uninstall.exe on the device."
        $script:uninstallPresent = $true
    }

    writeToLog V "uninstallPresent variable is set to: $uninstallPresent"

    writeToLog V "exeMissing evaluation complete."
    writeToLog V "New state of the exeMissing variable after evaluation: $exeMissing"

    writeToLog V ("Completed running {0} function." -f $MyInvocation.MyCommand)
}

function validateServices() {
    writeToLog V ("Started running {0} function." -f $MyInvocation.MyCommand)

    $tcService = "BASupportExpressStandaloneService_LOGICnow"
    $tcuService = "BASupportExpressSrvcUpdater_LOGICnow"

    $script:serviceMissing = $false
    $script:serviceStopped = $false

    writeToLog V "serviceMissing variable is set to: $serviceMissing"
    writeToLog V "serviceStopped variable is set to: $serviceStopped"
    writeToLog V "Attempting to get status of the Take Control services."

    try {
        $tcServiceStatus = (Get-Service $tcService -ErrorAction Stop).Status
        writeToLog I "Successfully detected the Take Control Agent service on the device."
        writeToLog I "Detected the state of the Take Control Agent service as: $tcServiceStatus"
        
        If ($tcServiceStatus -ne "Running") {
            writeToLog W "The Take Control Agent service is currently not running."
            $serviceStopped = $true
        }
    } catch {
        $msg = $_.Exception.Message
        writeToLog W "Was unable to detect the Take Control Agent service on the device due to:`r`n$msg"
        $serviceMissing = $true
    }

    try {
        $tcuServiceStatus = (Get-Service $tcuService -ErrorAction Stop).Status
        writeToLog I "Successfully detected the Take Control Agent Updater service on the device."
        writeToLog I "Detected the state of the Take Control Agent Updater service as: $tcuServiceStatus"

        If ($tcuServiceStatus -ne "Running") {
            writeToLog W "The Take Control Agent Updater service is currently not running."
        }
    } catch {
        $msg = $_.Exception.Message
        writeToLog W "Was unable to detect the Take Control Agent Updater service on the device due to:`r`n$msg"
        $serviceMissing = $true

        If ($tcuServiceStatus -ne "Running") {
            writeToLog W "The Take Control Agent service is currently not running."
            $serviceStopped = $true
        }
    }
    
    writeToLog V "serviceMissing evaluation complete."
    writeToLog V "New state of the serviceMissing variable after evaluation: $serviceMissing"

    writeToLog V "serviceStopped evaluation complete."
    writeToLog V "New state of the serviceStopped variable after evaluation: $serviceStopped"

    writeToLog V ("Completed running {0} function." -f $MyInvocation.MyCommand)
}

function detectRegistry() {
    writeToLog V ("Started running {0} function." -f $MyInvocation.MyCommand)

    $keyLocation = "HKLM:\SOFTWARE\WOW6432Node\Multiplicar Negocios\BeAnyWhere Support Express\GetSupportService_LOGICnow"

    $regKeyTest = Test-Path $keyLocation
    $script:regMissing = $false

    writeToLog V "regMissing variable is set to: $regMissing"

    writeToLog V "Testing if Take Control registry key is present."

    If ($regKeyTest -eq $false) {
        writeToLog W "Did not locate the registry key on the device."
        $script:regMissing = $true
    } Else {
        writeToLog I "Successfully detected registry key."
    }

    writeToLog V "regMissing evaluation complete."
    writeToLog V "New state of the regMissing variable after evaluation: $regMissing"

    writeToLog V ("Completed running {0} function." -f $MyInvocation.MyCommand)
}

function evaluateState() {
    writeToLog V ("Started running {0} function." -f $MyInvocation.MyCommand)

    $script:removalRequired = $false

    switch ($true) {
        $lockPresent {
            writeToLog W "Install / uninstall lock file present on the device."
            $script:removalRequired = $true
        }
        $exeMissing {
            writeToLog W "Device is missing a core exe component(s) to function correctly."
            $script:removalRequired = $true
        }
        $serviceMissing {
            writeToLog W "Device does not have one or more of the core services on the device."
            $script:removalRequired = $true
        }
        $serviceStopped {
            writeToLog W "Device has the services present, but currently are not running."
            $script:removalRequired = $true
        }
        $regMissing {
            writeToLog W "Registry key is missing on the device."
            $script:removalRequired = $true
        }
    }

    writeToLog V "removalRequired evaluation complete."
    writeToLog V "removalRequired variable new state after evaluation is now: $removalRequired"

    If ($removalRequired -eq $true) {
        writeToLog W "One or more issues have been detected with the Take Control application."
        writeToLog W "As such, a cleanup is required."
    } Else {
        writeToLog I "Determined Take Control as functional."
        writeToLog I "A Take Control cleanup is not required for this device."
    }

    writeToLog V ("Completed running {0} function." -f $MyInvocation.MyCommand)
}

function evaluateCleanup() {
    writeToLog V ("Started running {0} function." -f $MyInvocation.MyCommand)

    If ($removalRequired -eq $true) {
        writeToLog W "A cleanup is required of Take Control Agent."

        If ($uninstallPresent -eq $true) {
            
            $switches = "/S"
            $pinfo = New-Object System.Diagnostics.ProcessStartInfo
            $pinfo.FileName = $uninstallLocation
            $pinfo.RedirectStandardError = $true
            $pinfo.RedirectStandardOutput = $true
            $pinfo.UseShellExecute = $false
            $pinfo.Arguments = $switches

            writeToLog V "Using switch: $switches"
            writeToLog V "Running the following uninstaller:`r`n$uninstallLocation"

            $p = New-Object System.Diagnostics.Process
            $p.StartInfo = $pinfo
            $p.Start() | Out-Null
            $p.WaitForExit()
            $script:exitCode = $p.ExitCode
            
            If ($exitCode -ne 0) {
                writeToLog W "Exit Code returned:$exitCode"
            } ElseIf ($exitCode -eq 0) {
                writeToLog I "Successfully returned exitcode 0 from uninstall."
            }
        } Else {
            writeToLog V "Uninstall.exe was not present, so will only be able to perform manual cleanup."
        }

        writeToLog V "Now performing manual cleanup."
        If ($regMissing -eq $false) {
            try {
                Remove-Item $keyLocation -Force -Recurse -ErrorAction SilentlyContinue
            }
            catch {
                $msg = $_.Exception.Message
                writeToLog W "Was unable to perform cleanup on registry, due to:`r`n$msg"
            }
        }

        try {
            Remove-Item $installLockFile -Force -ErrorAction SilentlyContinue
            Remove-Item $uninstallLockFile -Force -ErrorAction SilentlyContinue
            Remove-Item $installLocation -Force -Recurse -ErrorAction SilentlyContinue
            Remove-Item $prgDataLocation -Force -Recurse -ErrorAction SilentlyContinue
        }
        catch {
            $msg = $_.Exception.Message
            writeToLog W "Was unable to perform cleanup, due to:`r`n$msg"
        }
        
    } Else {
        writeToLog I "Device does not require a cleanup. Take Control should be functional on the device."
    }

    writeToLog V ("Completed running {0} function." -f $MyInvocation.MyCommand)
}

function getTimeStamp() {
    return "[{0:dd/MM/yy} {0:HH:mm:ss}]" -f (Get-Date)
}

function writeToLog($state, $message) {

	switch -regex -Wildcard ($state) {
		"I" {
			$state = "INFO"
		}
		"E" {
			$state = "ERROR"
		}
		"W" {
			$state = "WARNING"
		}
		"F"  {
			$state = "FAILURE"
        }
        "V"  {
            If ($verboseMode -eq $true) {
                $state = "VERBOSE"
            } Else {
                return
            }
		}
		""  {
			$state = "INFO"
		}
		Default {
			$state = "INFO"
		}
     }

    Write-Host "$(getTimeStamp) - [$state]: $message"
    Write-Output "$(getTimeStamp) - [$state]: $message" | Out-file $logFile -Append
}

function main() {
    setupLogging
    validateUserInput
    validateFileLocks
    validateEXEs
    validateServices
    detectRegistry
    evaluateState
    evaluateCleanup
}
main